// Get the browser API (Chrome or Firefox)
const browserAPI = typeof browser !== 'undefined' ? browser : chrome;

// Store all tabs for filtering
let allTabs = [];
let filteredTabsList = [];
let selectedTabIndex = -1;
let groupMode = 'none'; // 'none', 'hostname', or 'category'
let sortGroupsAscending = true;
let filterAudioTabs = false;
let categories = {}; // Store categories: { "categoryName": ["domain1.com", "domain2.com"] }

// Remove protocol from URL for matching (https://, http://, etc.)
function removeProtocol(url) {
  if (!url) return '';
  return url.replace(/^https?:\/\//i, '').replace(/^\/\//, '');
}

// Extract hostname (including port) from URL - everything before the first slash
function extractHostname(url) {
  if (!url) return '';
  try {
    const urlObj = new URL(url);
    return urlObj.host; // host includes port if present (e.g., "example.com:8080")
  } catch (e) {
    // Fallback: extract manually
    const withoutProtocol = removeProtocol(url);
    const firstSlash = withoutProtocol.indexOf('/');
    if (firstSlash === -1) {
      return withoutProtocol;
    }
    return withoutProtocol.substring(0, firstSlash);
  }
}

// Group tabs by hostname
function groupTabsByHostname(tabs) {
  const groups = {};
  
  tabs.forEach(tab => {
    const hostname = extractHostname(tab.url || '');
    if (!groups[hostname]) {
      groups[hostname] = [];
    }
    groups[hostname].push(tab);
  });
  
  return groups;
}

// Group tabs by category, with uncategorized tabs in their original order
function groupTabsByCategory(tabs) {
  const groups = {};
  const uncategorized = [];
  
  // First pass: separate categorized and uncategorized tabs
  tabs.forEach(tab => {
    const hostname = extractHostname(tab.url || '');
    const categoryName = getCategoryForDomain(hostname);
    
    if (categoryName) {
      if (!groups[categoryName]) {
        groups[categoryName] = [];
      }
      groups[categoryName].push(tab);
    } else {
      uncategorized.push(tab);
    }
  });
  
  // Store uncategorized tabs as array (in original order)
  if (uncategorized.length > 0) {
    groups['Uncategorized'] = uncategorized;
  }
  
  return groups;
}

// Filter tabs based on URL match (excluding protocol) and audio
function filterTabs(tabs, filterText) {
  let filtered = tabs;
  
  // Filter by URL text if provided
  if (filterText && filterText.trim() !== '') {
    const filter = filterText.trim().toLowerCase();
    const urlWithoutProtocol = removeProtocol(filter);
    
    filtered = filtered.filter(tab => {
      const tabUrl = tab.url || '';
      const tabUrlWithoutProtocol = removeProtocol(tabUrl).toLowerCase();
      
      // Match if the filter text appears in the URL (without protocol)
      return tabUrlWithoutProtocol.includes(urlWithoutProtocol);
    });
  }
  
  // Filter by audio if audio filter is active
  if (filterAudioTabs) {
    filtered = filtered.filter(tab => tab.audible === true);
  }
  
  return filtered;
}

// Display a single tab item (returns DOM element)
function renderTabItem(tab, index) {
  const favicon = tab.favIconUrl || 'data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16"><rect width="16" height="16" fill="%23ccc"/></svg>';
  const title = tab.title || 'Untitled';
  const url = tab.url || '';
  const isActive = tab.active === true;
  const isChild = tab.openerTabId !== undefined;
  const isAudible = tab.audible === true;
  
  const tabItem = document.createElement('div');
  tabItem.className = 'tab-item';
  if (isActive) tabItem.classList.add('tab-item-active');
  if (isChild) tabItem.classList.add('tab-item-child');
  tabItem.setAttribute('data-tab-id', tab.id);
  tabItem.setAttribute('data-index', index);
  tabItem.setAttribute('data-opener-tab-id', tab.openerTabId || '');
  tabItem.setAttribute('role', 'option');
  
  const faviconImg = document.createElement('img');
  faviconImg.src = favicon;
  faviconImg.alt = '';
  faviconImg.className = 'tab-favicon';
  faviconImg.addEventListener('error', () => {
    faviconImg.style.display = 'none';
  });
  tabItem.appendChild(faviconImg);
  
  if (isAudible) {
    const audioIndicator = document.createElement('span');
    audioIndicator.className = 'audio-indicator';
    audioIndicator.textContent = '🔊';
    tabItem.appendChild(audioIndicator);
  }
  
  const tabContent = document.createElement('div');
  tabContent.className = 'tab-content';
  
  const tabTitle = document.createElement('div');
  tabTitle.className = 'tab-title';
  tabTitle.title = title;
  tabTitle.textContent = title;
  tabContent.appendChild(tabTitle);
  
  const tabUrl = document.createElement('div');
  tabUrl.className = 'tab-url';
  tabUrl.title = url;
  tabUrl.textContent = url;
  tabContent.appendChild(tabUrl);
  
  tabItem.appendChild(tabContent);
  
  const killButton = document.createElement('button');
  killButton.className = 'kill-tab-button';
  killButton.setAttribute('data-tab-id', tab.id);
  killButton.title = 'Close tab';
  killButton.textContent = '×';
  tabItem.appendChild(killButton);
  
  return tabItem;
}

// Create a group header element
function createGroupHeader(displayName, count, hostname, categoryName) {
  const header = document.createElement('div');
  header.className = 'group-header';
  
  const hostnameSpan = document.createElement('span');
  hostnameSpan.className = 'group-hostname';
  hostnameSpan.textContent = displayName;
  header.appendChild(hostnameSpan);
  
  const countSpan = document.createElement('span');
  countSpan.className = 'group-count';
  countSpan.textContent = count;
  header.appendChild(countSpan);
  
  if (hostname) {
    const addCategoryButton = document.createElement('button');
    addCategoryButton.className = 'add-category-button';
    addCategoryButton.setAttribute('data-hostname', hostname);
    addCategoryButton.title = 'Add to category';
    addCategoryButton.textContent = '📁';
    header.appendChild(addCategoryButton);
  }
  
  const killGroupButton = document.createElement('button');
  killGroupButton.className = 'kill-group-button';
  if (hostname) {
    killGroupButton.setAttribute('data-hostname', hostname);
  }
  if (categoryName) {
    killGroupButton.setAttribute('data-category', categoryName);
  }
  killGroupButton.title = 'Close all tabs in this group';
  killGroupButton.textContent = '×';
  header.appendChild(killGroupButton);
  
  return header;
}

// Display tabs in the list
function displayTabs(tabs) {
  const tabsList = document.getElementById('tabsList');
  filteredTabsList = tabs;
  selectedTabIndex = -1;
  
  // Clear existing content
  tabsList.textContent = '';
  
  if (tabs.length === 0) {
    const emptyDiv = document.createElement('div');
    emptyDiv.className = 'empty';
    emptyDiv.textContent = 'No matching tabs';
    tabsList.appendChild(emptyDiv);
    return;
  }
  
  let globalIndex = 0;
  
  if (groupMode === 'hostname') {
    // Group tabs by hostname
    const groups = groupTabsByHostname(tabs);
    
    // Sort groups by count
    const sortedHostnames = Object.keys(groups).sort((a, b) => {
      const countA = groups[a].length;
      const countB = groups[b].length;
      if (sortGroupsAscending) {
        return countA - countB;
      } else {
        return countB - countA;
      }
    });
    
    sortedHostnames.forEach(hostname => {
      const tabCount = groups[hostname].length;
      const categoryName = getCategoryForDomain(hostname);
      // Add group header with count, category button, and delete button
      const displayName = categoryName ? `${hostname} (${categoryName})` : hostname;
      const header = createGroupHeader(displayName, tabCount, hostname, null);
      tabsList.appendChild(header);
      
      // Add tabs in this group
      groups[hostname].forEach(tab => {
        const tabElement = renderTabItem(tab, globalIndex);
        tabsList.appendChild(tabElement);
        globalIndex++;
      });
    });
  } else if (groupMode === 'category') {
    // Group tabs by category
    const groups = groupTabsByCategory(tabs);
    
    // Sort groups by count
    const sortedCategories = Object.keys(groups).sort((a, b) => {
      const countA = groups[a].length;
      const countB = groups[b].length;
      if (sortGroupsAscending) {
        return countA - countB;
      } else {
        return countB - countA;
      }
    });
    
    sortedCategories.forEach(categoryName => {
      const categoryTabs = groups[categoryName];
      
      // Regular category group (array of tabs)
      const tabCount = categoryTabs.length;
      const header = createGroupHeader(categoryName, tabCount, null, categoryName);
      tabsList.appendChild(header);
      
      // Add tabs in this category (in their original order)
      categoryTabs.forEach(tab => {
        const tabElement = renderTabItem(tab, globalIndex);
        tabsList.appendChild(tabElement);
        globalIndex++;
      });
    });
  } else {
    // Display tabs normally (no grouping)
    tabs.forEach((tab, index) => {
      const tabElement = renderTabItem(tab, index);
      tabsList.appendChild(tabElement);
    });
  }
  
  // Add click handlers to switch to tabs
  document.querySelectorAll('.tab-item').forEach(item => {
    const tabContent = item.querySelector('.tab-content');
    tabContent.addEventListener('click', () => {
      const tabId = parseInt(item.dataset.tabId);
      activateTab(tabId);
    });
  });
  
  // Add click handlers to kill individual tabs
  document.querySelectorAll('.kill-tab-button').forEach(button => {
    button.addEventListener('click', (e) => {
      e.stopPropagation();
      const tabId = parseInt(button.dataset.tabId);
      killTab(tabId);
    });
  });
  
  // Add click handlers to kill all tabs in a group
  document.querySelectorAll('.kill-group-button').forEach(button => {
    button.addEventListener('click', (e) => {
      e.stopPropagation();
      const hostname = button.dataset.hostname;
      const category = button.dataset.category;
      killGroupTabs(hostname || category);
    });
  });
  
  // Add click handlers for category buttons
  document.querySelectorAll('.add-category-button').forEach(button => {
    button.addEventListener('click', async (e) => {
      e.stopPropagation();
      const hostname = button.dataset.hostname;
      await handleAddToCategory(hostname);
    });
  });
  
  // Update selected tab highlight
  updateSelectedTab();
}

// Activate a tab by ID
function activateTab(tabId) {
  browserAPI.tabs.update(tabId, { active: true });
  window.close();
}

// Kill a single tab by ID
function killTab(tabId) {
  browserAPI.tabs.remove(tabId);
  // Reload tabs to update the list
  loadTabs();
}

// Kill all filtered tabs
function killAllFilteredTabs() {
  if (filteredTabsList.length === 0) return;
  
  const tabIds = filteredTabsList.map(tab => tab.id);
  browserAPI.tabs.remove(tabIds);
  // Reload tabs to update the list
  loadTabs();
}

// Kill all tabs in a specific group (by hostname or category)
function killGroupTabs(hostnameOrCategory) {
  let groupTabs = [];
  
  if (groupMode === 'category') {
    // Kill by category
    const groups = groupTabsByCategory(filteredTabsList);
    groupTabs = groups[hostnameOrCategory] || [];
  } else {
    // Kill by hostname (existing logic)
    const groups = groupTabsByHostname(filteredTabsList);
    groupTabs = groups[hostnameOrCategory] || [];
  }
  
  if (groupTabs.length === 0) return;
  
  const tabIds = groupTabs.map(tab => tab.id);
  browserAPI.tabs.remove(tabIds);
  // Reload tabs to update the list
  loadTabs();
}

// Update visual highlight for selected tab
function updateSelectedTab() {
  document.querySelectorAll('.tab-item').forEach((item, index) => {
    if (index === selectedTabIndex) {
      item.classList.add('tab-item-selected');
      // Scroll into view
      item.scrollIntoView({ block: 'nearest' });
    } else {
      item.classList.remove('tab-item-selected');
    }
  });
}

// Handle keyboard navigation in tabs list
function handleTabsListKeydown(e) {
  const tabItems = document.querySelectorAll('.tab-item');
  if (tabItems.length === 0) return;
  
  switch (e.key) {
    case 'ArrowDown':
      e.preventDefault();
      selectedTabIndex = (selectedTabIndex + 1) % tabItems.length;
      updateSelectedTab();
      break;
      
    case 'ArrowUp':
      e.preventDefault();
      selectedTabIndex = selectedTabIndex <= 0 ? tabItems.length - 1 : selectedTabIndex - 1;
      updateSelectedTab();
      break;
      
    case 'Enter':
      e.preventDefault();
      if (selectedTabIndex >= 0 && selectedTabIndex < tabItems.length) {
        const tabId = parseInt(tabItems[selectedTabIndex].dataset.tabId);
        activateTab(tabId);
      }
      break;
      
    case 'Tab':
      // Allow Tab to move focus away from tabs list
      if (e.shiftKey) {
        // Shift+Tab goes back to filter input
        e.preventDefault();
        document.getElementById('urlFilter').focus();
      }
      // Regular Tab allows default behavior to move to next focusable element
      break;
  }
}

// Get all open tabs and display them
async function loadTabs() {
  try {
    allTabs = await new Promise((resolve) => {
      browserAPI.tabs.query({}, resolve);
    });
    
    if (allTabs.length === 0) {
      const tabsList = document.getElementById('tabsList');
      tabsList.textContent = '';
      const emptyDiv = document.createElement('div');
      emptyDiv.className = 'empty';
      emptyDiv.textContent = 'No open tabs';
      tabsList.appendChild(emptyDiv);
      return;
    }
    
    // Apply current filter if any
    const filterInput = document.getElementById('urlFilter');
    const filteredTabs = filterTabs(allTabs, filterInput ? filterInput.value : '');
    displayTabs(filteredTabs);
  } catch (error) {
    console.error('Error loading tabs:', error);
    const tabsList = document.getElementById('tabsList');
    tabsList.textContent = '';
    const errorDiv = document.createElement('div');
    errorDiv.className = 'error';
    errorDiv.textContent = 'Error loading tabs. Please try again.';
    tabsList.appendChild(errorDiv);
  }
}

// Handle filter input
function setupFilter() {
  const filterInput = document.getElementById('urlFilter');
  const tabsList = document.getElementById('tabsList');
  
  filterInput.addEventListener('input', (e) => {
    const filterText = e.target.value;
    const filteredTabs = filterTabs(allTabs, filterText);
    displayTabs(filteredTabs);
    // Reset selection when filtering
    selectedTabIndex = -1;
  });
  
  // Re-apply filters when audio filter changes (handled in setupAudioFilter)
  
  // Handle Tab key in filter input - move to tabs list
  filterInput.addEventListener('keydown', (e) => {
    if (e.key === 'Tab' && !e.shiftKey) {
      // Tab moves to tabs list
      const tabItems = document.querySelectorAll('.tab-item');
      if (tabItems.length > 0) {
        e.preventDefault();
        tabsList.focus();
        selectedTabIndex = 0;
        updateSelectedTab();
      }
    }
  });
  
  // Focus the filter input when popup opens
  filterInput.focus();
}

// Setup keyboard navigation for tabs list
function setupTabsListNavigation() {
  const tabsList = document.getElementById('tabsList');
  
  tabsList.addEventListener('keydown', handleTabsListKeydown);
  
  // When tabs list receives focus, select first tab if none selected
  tabsList.addEventListener('focus', () => {
    const tabItems = document.querySelectorAll('.tab-item');
    if (tabItems.length > 0 && selectedTabIndex < 0) {
      selectedTabIndex = 0;
      updateSelectedTab();
    }
  });
}

// Setup kill all button
function setupKillAllButton() {
  const killAllButton = document.getElementById('killAllButton');
  killAllButton.addEventListener('click', (e) => {
    e.stopPropagation();
    killAllFilteredTabs();
  });
}

// Setup group toggle button
function setupGroupToggle() {
  const groupToggle = document.getElementById('groupToggle');
  const sortToggle = document.getElementById('sortToggle');
  
  // Update button text based on current mode
  function updateButtonText() {
    if (groupMode === 'none') {
      groupToggle.textContent = 'Group';
      sortToggle.style.display = 'none';
      groupToggle.classList.remove('group-toggle-active');
    } else if (groupMode === 'hostname') {
      groupToggle.textContent = 'Hostname';
      sortToggle.style.display = 'flex';
      groupToggle.classList.add('group-toggle-active');
    } else if (groupMode === 'category') {
      groupToggle.textContent = 'Category';
      sortToggle.style.display = 'flex';
      groupToggle.classList.add('group-toggle-active');
    }
  }
  
  groupToggle.addEventListener('click', () => {
    // Cycle through modes: none -> hostname -> category -> none
    if (groupMode === 'none') {
      groupMode = 'hostname';
    } else if (groupMode === 'hostname') {
      groupMode = 'category';
    } else {
      groupMode = 'none';
    }
    
    updateButtonText();
    
    // Re-display tabs with new grouping
    const filterInput = document.getElementById('urlFilter');
    const filteredTabs = filterTabs(allTabs, filterInput.value);
    displayTabs(filteredTabs);
  });
  
  // Setup sort toggle button
  if (sortToggle) {
    // Set initial state
    if (sortGroupsAscending) {
      sortToggle.classList.add('sort-toggle-ascending');
      sortToggle.textContent = '↑';
    } else {
      sortToggle.classList.add('sort-toggle-descending');
      sortToggle.textContent = '↓';
    }
    
    sortToggle.addEventListener('click', () => {
      sortGroupsAscending = !sortGroupsAscending;
      
      // Update button appearance
      if (sortGroupsAscending) {
        sortToggle.classList.add('sort-toggle-ascending');
        sortToggle.classList.remove('sort-toggle-descending');
        sortToggle.textContent = '↑';
      } else {
        sortToggle.classList.add('sort-toggle-descending');
        sortToggle.classList.remove('sort-toggle-ascending');
        sortToggle.textContent = '↓';
      }
      
      // Re-display tabs with new sorting
      const filterInput = document.getElementById('urlFilter');
      const filteredTabs = filterTabs(allTabs, filterInput.value);
      displayTabs(filteredTabs);
    });
  }
  
  // Initialize button state
  updateButtonText();
}

// Setup audio filter toggle button
function setupAudioFilter() {
  const audioFilterToggle = document.getElementById('audioFilterToggle');
  
  if (audioFilterToggle) {
    audioFilterToggle.addEventListener('click', () => {
      filterAudioTabs = !filterAudioTabs;
      
      // Update button appearance
      if (filterAudioTabs) {
        audioFilterToggle.classList.add('audio-filter-active');
      } else {
        audioFilterToggle.classList.remove('audio-filter-active');
      }
      
      // Re-display tabs with new filter
      const filterInput = document.getElementById('urlFilter');
      const filteredTabs = filterTabs(allTabs, filterInput.value);
      displayTabs(filteredTabs);
    });
  }
}

// Load categories from storage
async function loadCategories() {
  try {
    const result = await new Promise((resolve) => {
      browserAPI.storage.sync.get(['categories'], resolve);
    });
    categories = result.categories || {};
  } catch (error) {
    console.error('Error loading categories:', error);
    categories = {};
  }
}

// Get category name for a domain
function getCategoryForDomain(hostname) {
  for (const [categoryName, domains] of Object.entries(categories)) {
    if (domains.includes(hostname)) {
      return categoryName;
    }
  }
  return null;
}

// Add domain to category
async function addDomainToCategory(hostname, categoryName) {
  if (!categories[categoryName]) {
    categories[categoryName] = [];
  }
  
  if (!categories[categoryName].includes(hostname)) {
    categories[categoryName].push(hostname);
    
    try {
      await new Promise((resolve) => {
        browserAPI.storage.sync.set({ categories: categories }, resolve);
      });
      return true;
    } catch (error) {
      console.error('Error saving category:', error);
      return false;
    }
  }
  return true; // Already in category
}

// Handle add to category button click
async function handleAddToCategory(hostname) {
  // Get list of existing categories
  const categoryNames = Object.keys(categories);
  
  // Show prompt to select or create category
  let categoryName = '';
  
  if (categoryNames.length > 0) {
    // Show selection dialog
    const selected = prompt(
      `Add "${hostname}" to category:\n\n` +
      `Existing categories: ${categoryNames.join(', ')}\n\n` +
      `Enter category name (or leave empty to cancel):`
    );
    
    if (selected === null) return; // User cancelled
    
    categoryName = selected.trim();
  } else {
    // No categories yet, prompt for new category name
    categoryName = prompt(`Create new category for "${hostname}":\n\nEnter category name:`);
    if (!categoryName) return; // User cancelled or empty
    categoryName = categoryName.trim();
  }
  
  if (categoryName) {
    const success = await addDomainToCategory(hostname, categoryName);
    if (success) {
      // Reload categories and refresh display
      await loadCategories();
      const filterInput = document.getElementById('urlFilter');
      const filteredTabs = filterTabs(allTabs, filterInput ? filterInput.value : '');
      displayTabs(filteredTabs);
    }
  }
}

// Refresh display when categories change
async function refreshDisplay() {
  await loadCategories();
  const filterInput = document.getElementById('urlFilter');
  const filteredTabs = filterTabs(allTabs, filterInput ? filterInput.value : '');
  displayTabs(filteredTabs);
}

// Load tabs when popup opens
document.addEventListener('DOMContentLoaded', async () => {
  await loadCategories();
  loadTabs();
  setupFilter();
  setupTabsListNavigation();
  setupKillAllButton();
  setupGroupToggle();
  setupAudioFilter();
  
  // Listen for storage changes (when categories are updated in settings)
  browserAPI.storage.onChanged.addListener((changes, areaName) => {
    if (areaName === 'sync' && changes.categories) {
      refreshDisplay();
    }
  });
  
  // Also focus after a short delay to ensure it gets focus even if tabs are still loading
  setTimeout(() => {
    const filterInput = document.getElementById('urlFilter');
    if (filterInput) {
      filterInput.focus();
    }
  }, 100);
});
